#!/usr/bin/env cwl-runner
cwlVersion: v1.2
class: Workflow
label: BD Rhapsody™ Sequence Analysis Pipeline
doc: >-
  The BD Rhapsody™ assays are used to create sequencing libraries from single
  cell transcriptomes.


  After sequencing, the analysis pipeline takes the FASTQ files and a reference
  file for gene alignment. The pipeline generates molecular counts per cell,
  read counts per cell, metrics, and an alignment file.

requirements:
- class: ScatterFeatureRequirement
- class: MultipleInputFeatureRequirement
- class: SubworkflowFeatureRequirement
- class: StepInputExpressionRequirement
- class: InlineJavascriptRequirement

inputs:
  Exact_Cell_Count:
    type: int?
    label: Exact Cell Count
    doc: |-
      Set a specific number (>=1) of cells as putative, based on those with the highest error-corrected read count
  Expected_Cell_Count:
    type: int?
    label: Expected Cell Count
    doc: |-
      Optional.  Guide the basic putative cell calling algorithm by providing an estimate of the number of cells expected.  Usually this can be the number of cells loaded into the Rhapsody cartridge.  If there are multiple inflection points on the second derivative cumulative curve, this will ensure the one selected is near the expected.
  Exclude_Intronic_Reads:
    type: boolean?
    label: Exclude Intronic Reads
    doc: |-
      By default, reads aligned to exons and introns are considered and represented in molecule counts. Including intronic reads may increase sensitivity, resulting in an increase in molecule counts and the number of genes per cell for both cellular and nuclei samples. Intronic reads may indicate unspliced mRNAs and are also useful, for example, in the study of nuclei and RNA velocity. When set to true, intronic reads will be excluded.
  Generate_Bam:
    type: boolean?
    label: Generate Bam Output
    doc: >
      Default: false.  A Bam read alignment file contains reads from all the input libraries, but creating it can consume a lot of compute and disk resources. By setting this field to true, the Bam file will be created. This option is shared for both Bioproduct and ATAC libraries.
  Long_Reads:
    type: boolean?
    label: Long Reads (>=650bp)
    doc: >
      By default, we detect if there are any reads longer than 650bp and then flag QualCLAlign to use STARlong instead of STAR. This flag can be explicitly set if it is known in advance that there are reads longer than 650bp.
  Custom_STAR_Params:
    type: string?
    label: Custom STAR Params
    doc: >
      Allows you to specify custom STAR aligner mapping parameters. Only the mapping parameters you provide here will be used with STAR, meaning that you must provide the complete list of parameters that you want to take effect.
      For reference, the parameters used by default in the pipeline are:

          1. Short Reads: --outFilterScoreMinOverLread 0 --outFilterMatchNminOverLread 0 --outFilterMultimapScoreRange 0 --clip3pAdapterSeq AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA --seedSearchStartLmax 50 --outFilterMatchNmin 25 --limitOutSJcollapsed 2000000
          2. Long Reads: Same options as short reads + --seedPerReadNmax 10000


      Example input: --alignIntronMax 500000 --outFilterScoreMinOverLread 0 --limitOutSJcollapsed 2000000


      Important:
        1. This applies to fastqs provided in the Reads user input
        2. Please do not specify any non-mapping related params like: --runThreadN, --genomeDir --outSAMtype, etc.
        3. Please only use params supported by STAR version 2.7.10b
  Custom_bwa_mem2_Params:
    type: string?
    label: Custom bwa-mem2 Params
    doc: >
      Allows you to specify custom bwa-mem2 mapping parameters. Only the mapping parameters you provide here will be used with bwa-mem2, meaning that you must provide the complete list of parameters that you want to take effect.
      The pipeline uses program default mapping parameters.


      Example input: -k 15 -w 200 -r 2


      Important:
          1. This applies to fastqs provided in the Reads_ATAC user input
          2. Please do not specify any non-mapping related params like: -C, -t, etc.
          3. Please only use params supported by bwa-mem2 version 2.2.1
  Maximum_Threads:
    type: int?
    label: Maximum Number of Threads
    doc: |-
      The maximum number of threads to use in the pipeline. By default, all available cores are used.
  Reads:
    type: File[]?
    label: Reads
    doc: |-
      FASTQ files from libraries that may include WTA mRNA, Targeted mRNA, AbSeq, Sample Multiplexing, and related technologies
  Reads_ATAC:
    type: File[]?
    label: Reads-ATAC
    doc: |-
      FASTQ files from libraries generated using the ATAC assay protocol. Each lane of a library is expected to have 3 FASTQs - R1, R2 and I1/I2, where the index read contains the Cell Barcode and UMI sequence. Only applies to ATAC assays.
  Predefined_ATAC_Peaks:
    type: File?
    label: ATAC Predefined Peak Regions
    doc: |-
      An optional BED file containing pre-established chromatin accessibility peak regions for generating the ATAC cell-by-peak matrix. Only applies to ATAC assays.
  VDJ_Version:
    label: VDJ Species Version
    doc: The VDJ species and chain types.  This option should only be set for VDJ experiment.
    type:
    - "null"
    - type: enum
      name: 'VDJ_Version'
      symbols:
      - human
      - hs
      - mouse
      - mm
      - humanBCR
      - humanTCR
      - mouseBCR
      - mouseTCR
  Cell_Calling_Data:
    label: Cell Calling Data
    doc: |-
         Specify the data to be used for putative cell calling.
         The default data for putative cell calling will be determined the following way:
           - If mRNA and ATAC Reads exist, mRNA_and_ATAC is the default.
           - If only ATAC Reads exist, ATAC is the default.
           - Otherwise, mRNA is the default.
    type:
    - 'null'
    - type: enum
      name: 'Cell_Calling_Data'
      symbols:
      - mRNA
      - AbSeq
      - ATAC
      - mRNA_and_ATAC
      - VDJ
  Cell_Calling_ATAC_Algorithm:
    label: Cell Calling ATAC Algorithm
    doc: Specify the ATAC algorithm to be used for ATAC putative cell calling. The Basic algorithm is the default.
    type:
    - 'null'
    - type: enum
      name: 'Cell_Calling_ATAC_Algorithm'
      symbols:
      - Basic
      - Refined
  Cell_Calling_Bioproduct_Algorithm:
    label: Cell Calling Bioproduct Algorithm
    doc: Specify the bioproduct algorithm to be used for mRNA/AbSeq putative cell calling. The Basic algorithm is the default.
    type:
    - 'null'
    - type: enum
      name: 'Cell_Calling_Bioproduct_Algorithm'
      symbols:
      - Basic
      - Refined
  AbSeq_Reference:
    type: File[]?
    label: AbSeq Reference
  Targeted_Reference:
    type: File[]?
    label: Targeted Reference
  Reference_Archive:
    type: File?
    label: Reference Files Archive
  Supplemental_Reference:
   type: File[]?
   label: Supplemental Reference
  Run_Name:
    label: Run Name 
    type: string?
    doc: This is a name for output files, for example Experiment1_Metrics_Summary.csv. 
         Default if left empty is to name run based on a library. Any non-alpha numeric characters will be changed to a hyphen. 
  Tag_Names:
    type: string[]?
    label: Sample Tag Names
    doc: >
      Specify the Sample Tag number followed by - (hyphen) and a sample name to appear
      in the output files. For example: 4-Ramos. Should be alpha numeric, with + - and _ allowed.
      Any special characters: &, (), [], {}, <>, ?, | will be corrected to underscores. 
  Sample_Tags_Version:
    label: Sample Tags Version
    doc: The sample multiplexing kit version.  This option should only be set for a multiplexed experiment.
    type:
    - 'null'
    - type: enum
      name: 'Sample_Tags_Version'
      symbols:
      - human
      - hs
      - mouse
      - mm
      - flex
      - nuclei_includes_mrna
      - nuclei_atac_only
      - custom
  AbSeq_UMI:
    type: int?
  Write_Filtered_Reads:
    type: boolean?
  Target_analysis:
    type: boolean?
  VDJ_VGene_Evalue:
    type: float?
    label: e-value threshold for V gene
    doc: >
      The e-value threshold for V gene call by IgBlast, default is set as 0.001
  VDJ_JGene_Evalue:
    type: float?
    label: e-value threshold for J gene
    doc: >
      The e-value threshold for J gene call by IgBlast, default is set as 0.001
outputs:
  vdjCellsDatatable:
   label: vdjCellsDatatable
   type: File?
   outputSource: VDJ_Compile_Results/vdjCellsDatatable
  vdjCellsDatatableUncorrected:
   label: vdjCellsDatatableUncorrected
   type: File?
   outputSource: VDJ_Compile_Results/vdjCellsDatatableUncorrected
  vdjUnfilteredContigsAIRR:
   label: vdjUnfilteredContigsAIRR
   type: File?
   outputSource: VDJ_Compile_Results/vdjUnfilteredContigsAIRR
  vdjDominantContigsAIRR:
   label: vdjDominantContigsAIRR
   type: File?
   outputSource: VDJ_Compile_Results/vdjDominantContigsAIRR
  vdjMetricsCsv:
   label: vdjMetricsCsv
   type: File?
   outputSource: VDJ_Compile_Results/vdjMetricsCsv
  vdjDbecFilterImages:
   label: vdjDbecFilterImages
   type: File?
   outputSource: VDJ_Compile_Results/vdjDbecFilterImages
  Bioproduct_Stats:
    label: Bioproduct Statistics
    type: File?
    outputSource: GetDataTable/Bioproduct_Stats
  Bam:
    label: BAM files and indices
    type: File[]?
    outputSource: [MergeBAM_RNA/Bam, MergeBAM_RNA/BamIndex, MergeBAM_VDJ/Bam, MergeBAM_VDJ/BamIndex, MergeBAM_ATAC/Bam, MergeBAM_ATAC/BamIndex]
    linkMerge: merge_flattened
    pickValue: all_non_null
  Metrics_Summary:
    label: Metrics Summary
    type: File?
    outputSource: Metrics/Metrics_Summary
  Data_Tables:
    label: Data Tables
    type: File[]?
    outputSource: GetDataTable/Data_Tables
  Logs:
    label: Pipeline Logs
    type: Directory
    outputSource: BundleLogs/logs_dir
  Protein_Aggregates_Experimental:
    label: Protein Aggregates (Experimental)
    type: File?
    outputSource: GetDataTable/Protein_Aggregates_Experimental
  Multiplexing:
    type: File[]?
    outputSource: MergeMultiplex/Multiplex_out
  ATAC:
    type: File[]?
    outputSource: MergeATAC/ATAC_out
  Immune_Cell_Classification(Experimental):
    label: Immune Cell Classification (Experimental)
    type: File[]?
    outputSource: [GetDataTable/Cell_Type_Predictions, ATAC_Compile_Results/Cell_Type_Predictions]
    pickValue: all_non_null
    linkMerge: merge_flattened
  Dim_Reduction_Coord:
    label: Dimensionality Reduction Coordinates
    type: File[]?
    outputSource: [GetDataTable/Dim_Reduction_Coord, ATAC_Compile_Results/ATAC_Dim_Reduction_Coord]
    pickValue: all_non_null
    linkMerge: merge_flattened
  Pipeline_Report_HTML:
    label: Pipeline Report HTML
    type: File?
    outputSource: Metrics/Pipeline_Report_HTML
  H5MU:
    label: Cellismo File
    type: File?
    outputSource: GenerateH5MU/H5MU
  Seurat:
    label: Seurat RDS File
    type: File?
    outputSource: GenerateSeurat/SeuratRDS

  Failed_Reads_CSVs_RNA:
    type: File[]?
    outputSource: QualCLAlign_RNA/Failed_Reads_CSVs
  Failed_Reads_CSVs_ATAC:
    type: File[]?
    outputSource: QualCLAlign_ATAC/Failed_Reads_CSVs
steps:
  Start_Time:
    run:
      cwlVersion: v1.2
      class: ExpressionTool
      requirements:
      - class: InlineJavascriptRequirement
      inputs: []
      outputs:
        Start_Time:
          type: string
      expression: |-
       ${
         var today = new Date();
         var date = today.toDateString();
         var time = today.toLocaleTimeString('en-us', {timeZoneName: 'short'});
         return ({Start_Time: date + ' ' + time});
       } 
    in: []
    out:
      - Start_Time
  Version:
    run: Version.cwl
    in: [] 
    out: 
      - version
  Name_Settings:
    label: Name Settings
    run: NameSettings.cwl 
    in:
      _Run_Name: Run_Name
      Bioproduct_Fastqs:
        source: Reads
        valueFrom:
          ${
            var fastqs = [];
            if(self) {
              for(var i = 0; i < self.length; i++) {
                fastqs.push(self[i].basename);
              }
            }
            return fastqs;
          }
      ATAC_Fastqs:
        source: Reads_ATAC
        valueFrom:
          ${
            var fastqs = [];
            if(self) {
              for(var i = 0; i < self.length; i++) {
                fastqs.push(self[i].basename);
              }
            }
            return fastqs;
          }
    out:
      - Run_Name
      - Run_Base_Name
  Multiplexing_Settings:
    label: Multiplexing Settings
    run: MultiplexingSettings.cwl
    in:
      _Sample_Tag_Names: Tag_Names
      _Sample_Tags_Version: Sample_Tags_Version
    out:
    - Sample_Tag_Names
    - Sample_Tags_Version
  Assay_Settings:
    run: Assay_Settings.cwl
    in:
      Targeted_Reference_Present:
        source: Targeted_Reference
        valueFrom:
            ${
              if (self && self.length > 0){
                return true;
              }
              else{
                return false;
              }
            }
      Reference_Archive_Present:
        source: Reference_Archive
        valueFrom:
            ${
              if (self){
                return true;
              }
              else{
                return false;
              }
            }
      AbSeq_Reference_Present:
        source: AbSeq_Reference
        valueFrom:
            ${
              if (self && self.length > 0){
                return true;
              }
              else{
                return false;
              }
            }
      Reads_RNA_Present:
        source: Reads
        valueFrom:
            ${
              if (self && self.length > 0) {
                return true;
              }
              else {
                return false;
              }
            }
      Reads_ATAC_Present:
        source: Reads_ATAC
        valueFrom:
            ${
              if (self && self.length > 0) {
                return true;
              }
              else {
                return false;
              }
            }
      Cell_Calling_Data:
        source: Putative_Cell_Calling_Settings/Cell_Calling_Data
    out:
    - Assay_RNA
    - Assay_ATAC
  GetMachineResources:
    label: Get Machine Resources
    run: GetMachineResources.cwl
    in: []
    out:
      - Total_CPUs_Avail
  VDJ_Settings:
    label: VDJ Settings
    run: VDJ_Settings.cwl
    in:
      _VDJ_Version: VDJ_Version
    out:
      - VDJ_Version
  Putative_Cell_Calling_Settings:
    label: Putative Cell Calling Settings
    run: PutativeCellSettings.cwl
    in:
      _Cell_Calling_ATAC_Algorithm: Cell_Calling_ATAC_Algorithm
      _Cell_Calling_Bioproduct_Algorithm: Cell_Calling_Bioproduct_Algorithm
      _Cell_Calling_Data: Cell_Calling_Data
      _Exact_Cell_Count: Exact_Cell_Count
      _Expected_Cell_Count: Expected_Cell_Count
      _Sample_Tags_Version: Multiplexing_Settings/Sample_Tags_Version
      _Reads_Present:
        source: Reads
        valueFrom:
            ${
              if (self && self.length > 0) {
                return true;
              }
              else {
                return false;
              }
            }
      _Reads_ATAC_Present:
        source: Reads_ATAC
        valueFrom:
            ${
              if (self && self.length > 0) {
                return true;
              }
              else {
                return false;
              }
            }
    out:
      - Cell_Calling_ATAC_Algorithm
      - Cell_Calling_Bioproduct_Algorithm
      - Cell_Calling_Data
      - Exact_Cell_Count
      - Expected_Cell_Count
  Misc_Settings:
    label: Miscellaneous Settings
    run: MiscSettings.cwl
    in:
      _Exclude_Intronic_Reads: Exclude_Intronic_Reads
      _Long_Reads: Long_Reads
      _Custom_STAR_Params: Custom_STAR_Params
      _Custom_bwa_mem2_Params: Custom_bwa_mem2_Params
    out:
      - Exclude_Intronic_Reads
      - Long_Reads
      - Custom_STAR_Params
      - Custom_bwa_mem2_Params
  Bam_Settings:
    label: Bam Settings
    run: BamSettings.cwl
    in:
      _Generate_Bam: Generate_Bam
    out:
      - Generate_Bam
  Internal_Settings:
    label: Internal Settings
    run: InternalSettings.cwl
    in:
      _AbSeq_UMI: AbSeq_UMI
      _Target_analysis: Target_analysis
      _VDJ_VGene_Evalue: VDJ_VGene_Evalue
      _VDJ_JGene_Evalue: VDJ_JGene_Evalue
    out: 
    - AbSeq_UMI
    - Target_analysis
    - VDJ_VGene_Evalue
    - VDJ_JGene_Evalue
  CheckReference:
    requirements:
      ResourceRequirement:
        ramMin: 10000
    run: CheckReference.cwl
    in:
      Targeted_Reference: Targeted_Reference
      Reference_Archive: Reference_Archive
      Supplemental_Reference: Supplemental_Reference
      AbSeq_Reference: AbSeq_Reference
      Assay:
        source: [Assay_Settings/Assay_RNA, Assay_Settings/Assay_ATAC]
        linkMerge: merge_flattened
        pickValue: all_non_null
      VDJ_Version: VDJ_Settings/VDJ_Version
      Sample_Tags_Version: Multiplexing_Settings/Sample_Tags_Version
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
      Predefined_ATAC_Peaks: Predefined_ATAC_Peaks
    out:
    - Index
    - Extra_Seqs
    - Full_Genes
    - output
    - Transcript_Length
    - GTF
    - Reference_FASTA
    - Target_Gene_Mapping
    - Transcription_Factor_Motif_PFM
    - Checked_Predefined_Peaks
    - Checked_Predefined_Peaks_Index
  QualCLAlign_RNA:
    run: QualCLAlign.cwl
    in:
      Reads: Reads
      Assay: Assay_Settings/Assay_RNA
      Run_Base_Name: Name_Settings/Run_Base_Name
      VDJ_Version: VDJ_Settings/VDJ_Version
      Index: CheckReference/Index
      Extra_Seqs: CheckReference/Extra_Seqs
      STAR_Params: Misc_Settings/Custom_STAR_Params
      Long_Reads: Misc_Settings/Long_Reads
      # For local/internal deployment this sets the Threads param such that:
      # If Maximum_Threads is specified
      #   Threads = Maximum_Threads
      # However, if Maximum_Threads is unspecified AND ATAC data is also present
      #   Threads =  Total_CPUs / 2 if Total_CPUs >= 48 else Total_CPUs
      # Note: minCores for the task is set appropriately based on this number.
      #---------------------------------------------------------
      Threads:
        source: [Maximum_Threads, GetMachineResources/Total_CPUs_Avail, Assay_Settings/Assay_ATAC]
        valueFrom:
          ${
            var [max_threads, total_cpus_avail, assay_atac] = self;

            if (max_threads != null) {
              return max_threads;
            }

            if (assay_atac == null) {
              return total_cpus_avail;
            }

            if (total_cpus_avail >= 48) {
              return parseInt(total_cpus_avail / 2);
            }
            else {
              return total_cpus_avail;
            }
          }
      Write_Filtered_Reads: Write_Filtered_Reads
    out:
    - Bead_Version
    - Libraries
    - ReadsList
    - BAMFiles
    - QualCLAlignMetrics
    - Logs
    - Failed_Reads_CSVs
    when: $(inputs.Assay == "WTA" || inputs.Assay == "Targeted")
  QualCLAlign_ATAC:
    run: QualCLAlign.cwl
    in:
      Reads: Reads_ATAC
      Predefined_ATAC_Peaks: CheckReference/Checked_Predefined_Peaks
      Assay: Assay_Settings/Assay_ATAC
      Run_Base_Name: Name_Settings/Run_Base_Name
      Index: CheckReference/Index
      Generate_Bam: Bam_Settings/Generate_Bam
      bwa_mem2_Params: Misc_Settings/Custom_bwa_mem2_Params
      # For local/internal deployment this sets the Threads param such that:
      # If Maximum_Threads is specified
      #   Threads = Maximum_Threads
      # However, if Maximum_Threads is unspecified AND RNA data is also present
      #   Threads =  Total_CPUs / 2 if Total_CPUs >= 48 else Total_CPUs
      # Note: minCores for the task is set appropriately based on this number.
      #---------------------------------------------------------
      Threads:
        source: [Maximum_Threads, GetMachineResources/Total_CPUs_Avail, Assay_Settings/Assay_RNA]
        valueFrom:
          ${
            var [max_threads, total_cpus_avail, assay_rna] = self;

            if (max_threads != null) {
              return max_threads;
            }

            if (assay_rna == null) {
              return total_cpus_avail;
            }

            if (total_cpus_avail >= 48) {
              return parseInt(total_cpus_avail / 2);
            }
            else {
              return total_cpus_avail;
            }
          }
      Write_Filtered_Reads: Write_Filtered_Reads
    out:
    - Bead_Version
    - Libraries
    - ReadsList
    - BAMFiles
    - Fragments
    - Fragments_Index
    - Transposase_Sites
    - Transposase_Sites_Index
    - Peaks
    - Peaks_Index
    - QualCLAlignMetrics
    - UnifiedMetrics
    - Logs
    - Failed_Reads_CSVs
    - Reference_Genome_Size
    - Reference_Genome_Contig_Lengths
    when: $(inputs.Assay == "ATAC")
  Metadata_Settings:
    run: Metadata.cwl
    in:
      Start_Time: Start_Time/Start_Time
      Bead_Version:
        source: [QualCLAlign_RNA/Bead_Version, QualCLAlign_ATAC/Bead_Version]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Pipeline_Name:
        valueFrom: "BD Rhapsody Sequence Analysis Pipeline"
      Run_Name: Name_Settings/Run_Name
      Run_Base_Name: Name_Settings/Run_Base_Name
      Pipeline_Version: Version/version
      Libraries:
        source: QualCLAlign_RNA/Libraries
        pickValue: all_non_null
      Libraries_ATAC:
        source: QualCLAlign_ATAC/Libraries
        pickValue: all_non_null
      Reads:
        source: QualCLAlign_RNA/ReadsList
        pickValue: all_non_null
      Reads_ATAC:
        source: QualCLAlign_ATAC/ReadsList
        pickValue: all_non_null
      Exact_Cell_Count: Putative_Cell_Calling_Settings/Exact_Cell_Count
      Expected_Cell_Count: Putative_Cell_Calling_Settings/Expected_Cell_Count
      Exclude_Intronic_Reads: Misc_Settings/Exclude_Intronic_Reads
      Assay:
        source: [Assay_Settings/Assay_RNA, Assay_Settings/Assay_ATAC]
        linkMerge: merge_flattened
        pickValue: all_non_null
      VDJ_Version: VDJ_Settings/VDJ_Version
      AbSeq_Reference: AbSeq_Reference
      Generate_Bam: Bam_Settings/Generate_Bam
      Targeted_Reference: Targeted_Reference
      Reference_Archive: Reference_Archive
      Supplemental_Reference: Supplemental_Reference
      Predefined_ATAC_Peaks: Predefined_ATAC_Peaks
      Cell_Calling_ATAC_Algorithm: Putative_Cell_Calling_Settings/Cell_Calling_ATAC_Algorithm
      Cell_Calling_Bioproduct_Algorithm: Putative_Cell_Calling_Settings/Cell_Calling_Bioproduct_Algorithm
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
      Sample_Tags_Version: Multiplexing_Settings/Sample_Tags_Version
      Sample_Tag_Names: Multiplexing_Settings/Sample_Tag_Names
      Long_Reads: Misc_Settings/Long_Reads
      Custom_STAR_Params: Misc_Settings/Custom_STAR_Params
      Custom_bwa_mem2_Params: Misc_Settings/Custom_bwa_mem2_Params
    out:
    - Run_Metadata
  ATAC_Cell_by_Peak:
    run: ATAC_Cell_by_Peak.cwl
    in:
      Fragments: QualCLAlign_ATAC/Fragments
      Transposase_Sites: QualCLAlign_ATAC/Transposase_Sites
      Peaks: QualCLAlign_ATAC/Peaks
      Assay: Assay_Settings/Assay_ATAC
      Run_Metadata: Metadata_Settings/Run_Metadata
      GTF: CheckReference/GTF
      Map_Gene_Name_ID:
        source: Assay_Settings/Assay_RNA
        valueFrom: $(self == null)
    out:
      - Initial_Seurat_RDS
      - ATAC_Cell_Calling_Data
      - Total_Fragment_Metrics
      - Gene_Name_ID_Json
      - output
    when: $(inputs.Assay == "ATAC")
  AlignmentAnalysis:
    run: AlignmentAnalysis.cwl
    in:
      Assay: Assay_Settings/Assay_RNA
      Transcript_Length: CheckReference/Transcript_Length
      Run_Metadata: Metadata_Settings/Run_Metadata
      ReadQualityMetrics: QualCLAlign_RNA/QualCLAlignMetrics
      R2_BAM: QualCLAlign_RNA/BAMFiles
      GTF: CheckReference/GTF
      Extra_Seqs: CheckReference/Extra_Seqs
      Exclude_Intronic_Reads: Misc_Settings/Exclude_Intronic_Reads
      Maximum_Threads: Maximum_Threads
    out:
    - Seq_Metrics
    - Annotated_Bam_Files
    - Sorted_Valid_Reads_CSV
    - Gene_Name_ID_Json
    - num_valid_ig_reads
    - num_valid_tcr_reads
    - validIgReads
    - validTcrReads
    - num_cell_estimate
    - num_bioproducts
    when: $(inputs.Assay == "WTA" || inputs.Assay == "Targeted")
  AnnotateMolecules:
    requirements:
      ResourceRequirement:
        ramMin: 32000
    run: AnnotateMolecules.cwl
    in:
      Assay: Assay_Settings/Assay_RNA
      Valids:
        source: AlignmentAnalysis/Sorted_Valid_Reads_CSV
        default: ["does_not_exist"] # scatter does not like null.
      AbSeq_UMI: Internal_Settings/AbSeq_UMI
      Run_Metadata: Metadata_Settings/Run_Metadata
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
      Sample_Tags_Version: Multiplexing_Settings/Sample_Tags_Version
    out:
    - Bioproduct_Stats_List
    - Cell_Biop_Summary_List
    - Corrected_Mols_List
    - Total_Molecules
    - Num_Molecules_Corrected
    - output
    scatter:
    - Valids
    when: $((inputs.Assay == "WTA" || inputs.Assay == "Targeted") && (inputs.Cell_Calling_Data != 6 || inputs.Sample_Tags_Version != null))
  SumMoleculesCorrected:
    run:
      class: ExpressionTool
      cwlVersion: v1.2
      hints:
        - class: ResourceRequirement
          coresMin: 1
          ramMin: 1024
      inputs:
        Assay:
          type: string
        values:
          type: int[]
      outputs:
        Num_Mols_Corrected:
          type: int
      expression: |
        $({
          Num_Mols_Corrected: inputs.values.reduce((a, b) => a + b, 0)
        })
    in:
      Assay: Assay_Settings/Assay_RNA
      values: AnnotateMolecules/Num_Molecules_Corrected
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
    out: [ Num_Mols_Corrected ]
    when: $( (inputs.Assay == "WTA" || inputs.Assay == "Targeted" ) && inputs.Cell_Calling_Data != 6 )
  CalcSeqSaturation:
    run: CalcSeqSaturation.cwl
    in:
      Corrected_Mols: AnnotateMolecules/Corrected_Mols_List
      Metrics_Tar: GetDataTable/Metrics_tar
      Putative_Cells: GetDataTable/Cell_Order
      Run_Base_Name: Name_Settings/Run_Base_Name
      Valid_Reads_CSV: AlignmentAnalysis/Sorted_Valid_Reads_CSV
      Assay: Assay_Settings/Assay_RNA
      Targeted_Reference_Present:
        source: Targeted_Reference
        valueFrom:
            ${
              if (self && self.length > 0){
                return true;
              }
              else{
                return false;
              }
            }
      Reference_Archive_Present:
        source: Reference_Archive
        valueFrom:
            ${
              if (self){
                return true;
              }
              else{
                return false;
              }
            }
      AbSeq_Reference_Present:
        source: AbSeq_Reference
        valueFrom:
            ${
              if (self && self.length > 0){
                return true;
              }
              else{
                return false;
              }
            }
      Num_Mols_Corrected: SumMoleculesCorrected/Num_Mols_Corrected
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
    out:
      - Seq_Saturation_Plot_Json
    when: $((inputs.Assay == "WTA" || inputs.Assay == "Targeted") && (inputs.Cell_Calling_Data != 5 && inputs.Cell_Calling_Data != 6))
  GetDataTable:
    run: GetDataTable.cwl
    in:
      Cell_Biop_Summary_List:
        source: AnnotateMolecules/Cell_Biop_Summary_List
        pickValue: all_non_null
      Bioproduct_Stats_List:
        source: AnnotateMolecules/Bioproduct_Stats_List
        pickValue: all_non_null
      Total_Molecules:
        source: AnnotateMolecules/Total_Molecules
        pickValue: all_non_null
      Seq_Metrics: AlignmentAnalysis/Seq_Metrics
      ATAC_Cell_Calling_Input: ATAC_Cell_by_Peak/ATAC_Cell_Calling_Data
      Full_Genes: CheckReference/Full_Genes
      Run_Metadata: Metadata_Settings/Run_Metadata
      Gene_Name_ID_Json:
        source: [AlignmentAnalysis/Gene_Name_ID_Json, ATAC_Cell_by_Peak/Gene_Name_ID_Json]
        pickValue: first_non_null
      num_cell_estimate: AlignmentAnalysis/num_cell_estimate
      num_bioproducts: AlignmentAnalysis/num_bioproducts
    out:
    - Metrics_tar
    - Bioproduct_Stats
    - Cell_Order
    - Cell_Order_Subsampled
    - Cell_Type_Predictions
    - Data_Tables
    - Dim_Reduction_Coord
    - output
    - Protein_Aggregates_Experimental
    - Putative_Cells_Origin
    - Biop_putative_data_table
  SampleTag:
    run: SampleTag.cwl
    in:
      Putative_Cells:
        source: [GetDataTable/Cell_Order, VDJ_Compile_Results/Cell_Order]
        pickValue: first_non_null
      Run_Metadata: Metadata_Settings/Run_Metadata
      Cell_Biop_Summary_List:
        source: AnnotateMolecules/Cell_Biop_Summary_List
        pickValue: all_non_null
      Gene_Name_ID_Json:
        source: [AlignmentAnalysis/Gene_Name_ID_Json, ATAC_Cell_by_Peak/Gene_Name_ID_Json]
        pickValue: first_non_null
      Full_Genes: CheckReference/Full_Genes
      num_cell_estimate: AlignmentAnalysis/num_cell_estimate
      num_bioproducts: AlignmentAnalysis/num_bioproducts
      Sample_Tags_Version: Multiplexing_Settings/Sample_Tags_Version
    out:
      - SampleTag_Metrics_Tar
      - SampleTag_Calls
      - SampleTag_CSVs
      - SampleTag_perTagZips
    when: $(inputs.Sample_Tags_Version != null)
  ATAC_Compile_Results:
    run: ATAC_Compile_Results.cwl
    in:
      Initial_Seurat_RDS: ATAC_Cell_by_Peak/Initial_Seurat_RDS
      Cell_Order: GetDataTable/Cell_Order
      Cell_Order_Subsampled: GetDataTable/Cell_Order_Subsampled
      Unified_Metrics: QualCLAlign_ATAC/UnifiedMetrics
      Total_Fragment_Metrics: ATAC_Cell_by_Peak/Total_Fragment_Metrics
      Fragments: QualCLAlign_ATAC/Fragments
      Assay: Assay_Settings/Assay_ATAC
      Run_Metadata: Metadata_Settings/Run_Metadata
      Gene_Name_ID_Json:
        source: [AlignmentAnalysis/Gene_Name_ID_Json, ATAC_Cell_by_Peak/Gene_Name_ID_Json]
        pickValue: first_non_null
      Input_Metrics_tar: GetDataTable/Metrics_tar
      Reference_Genome_Size: QualCLAlign_ATAC/Reference_Genome_Size
      Biop_putative_data_table: GetDataTable/Biop_putative_data_table
      Reference_Genome_Contig_Lengths: QualCLAlign_ATAC/Reference_Genome_Contig_Lengths
    out:
      - Metrics_tar
      - ATAC_Data_Tables
      - ATAC_Dim_Reduction_Coord
      - ATAC_Seurat_RDS
      - Cell_Type_Predictions
      - output
    when: $(inputs.Assay == "ATAC")
  VDJ_Analyze_Reads_IG:
    run: VDJ_Analyze_Reads_IG.cwl
    when: $(inputs.VDJ_Version != null && inputs.VDJ_Version != "humanTCR" && inputs.VDJ_Version != "mouseTCR")
    in:
      Valid_Reads_Fastq_IG: AlignmentAnalysis/validIgReads
      Num_Valid_Reads_IG: AlignmentAnalysis/num_valid_ig_reads
      VDJ_Version: VDJ_Settings/VDJ_Version
      Maximum_Threads: Maximum_Threads
    out:
    - gatheredCalls
  VDJ_Analyze_Reads_TCR:
    run: VDJ_Analyze_Reads_TCR.cwl
    when: $(inputs.VDJ_Version != null && inputs.VDJ_Version != "humanBCR" && inputs.VDJ_Version != "mouseBCR")
    in:
      Valid_Reads_Fastq_TCR: AlignmentAnalysis/validTcrReads
      Num_Valid_Reads_TCR: AlignmentAnalysis/num_valid_tcr_reads
      VDJ_Version: VDJ_Settings/VDJ_Version
      Maximum_Threads: Maximum_Threads
    out:
    - gatheredCalls
  VDJ_Compile_Results:
    run: VDJ_Compile_Results.cwl
    when: $(inputs.VDJ_Version != null)
    in:
      cellTypeMapping:
        source: [GetDataTable/Cell_Type_Predictions, ATAC_Compile_Results/Cell_Type_Predictions]
        linkMerge: merge_flattened
        pickValue: all_non_null
      putativeCells: GetDataTable/Cell_Order
      chainsToIgnore:
        valueFrom: $([])
      igCalls: VDJ_Analyze_Reads_IG/gatheredCalls
      tcrCalls: VDJ_Analyze_Reads_TCR/gatheredCalls
      evalueVgene: Internal_Settings/VDJ_VGene_Evalue
      evalueJgene: Internal_Settings/VDJ_JGene_Evalue
      metadata:  Metadata_Settings/Run_Metadata
      Seq_Metrics: AlignmentAnalysis/Seq_Metrics
      VDJ_Version: VDJ_Settings/VDJ_Version
    out:
    - vdjCellsDatatable
    - vdjCellsDatatableUncorrected
    - vdjDominantContigsAIRR
    - vdjUnfilteredContigsAIRR
    - vdjMetricsCsv
    - vdjDbecFilterImages
    - vdjMetricsTar
    - Cell_Order
  AddtoBam_RNA:
    run: AddtoBam.cwl
    in:
      Assay: Assay_Settings/Assay_RNA
      Bam:
        source: AlignmentAnalysis/Annotated_Bam_Files
        default: ["does_not_exist"] # scatter does not like null
      Cell_Order: GetDataTable/Cell_Order
      Corrected_Mols: AnnotateMolecules/Corrected_Mols_List
      SampleTag_Calls: SampleTag/SampleTag_Calls
      Run_Metadata:  Metadata_Settings/Run_Metadata
      Target_Gene_Mapping: CheckReference/Target_Gene_Mapping
      Generate_Bam: Bam_Settings/Generate_Bam
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
    when: $(inputs.Generate_Bam == true && inputs.Cell_Calling_Data != 6 && (inputs.Assay == "WTA" || inputs.Assay == "Targeted"))
    out:
    - Annotated_Bam
    - output
    scatter:
    - Bam
  AddtoBam_VDJ:
    run: AddtoBam.cwl
    in:
      Assay: Assay_Settings/Assay_RNA
      Bam:
        source: AlignmentAnalysis/Annotated_Bam_Files
        default: ["does_not_exist"] # scatter does not like null
      SampleTag_Calls: SampleTag/SampleTag_Calls
      Corrected_Mols: AnnotateMolecules/Corrected_Mols_List
      Cell_Order: VDJ_Compile_Results/Cell_Order
      Run_Metadata:  Metadata_Settings/Run_Metadata
      Target_Gene_Mapping: CheckReference/Target_Gene_Mapping
      Generate_Bam: Bam_Settings/Generate_Bam
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
    when: $(inputs.Generate_Bam == true && inputs.Cell_Calling_Data == 6 && (inputs.Assay == "WTA" || inputs.Assay == "Targeted"))
    out:
    - Annotated_Bam
    - output
    scatter:
    - Bam
  AddtoBam_ATAC:
    run: AddtoBam.cwl
    in:
      Assay: Assay_Settings/Assay_ATAC
      Bam:
        source: [QualCLAlign_ATAC/BAMFiles]
        default: ["does_not_exist"]
      Cell_Order: GetDataTable/Cell_Order
      SampleTag_Calls: SampleTag/SampleTag_Calls
      Run_Metadata:  Metadata_Settings/Run_Metadata
      Generate_Bam: Bam_Settings/Generate_Bam
    when: $(inputs.Generate_Bam == true && inputs.Assay == "ATAC")
    out:
    - Annotated_Bam
    - output
    scatter:
    - Bam
  MergeBAM_RNA:
    run: MergeBAM.cwl
    in:
      Run_Metadata: Metadata_Settings/Run_Metadata
      BamFiles: AddtoBam_RNA/Annotated_Bam
      Generate_Bam: Bam_Settings/Generate_Bam
      Assay: Assay_Settings/Assay_RNA
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
    when: $(inputs.Generate_Bam == true && inputs.Cell_Calling_Data != 6 && (inputs.Assay == "WTA" || inputs.Assay == "Targeted"))
    out:
    - Bam
    - BamIndex
    - log
  MergeBAM_VDJ:
    run: MergeBAM.cwl
    in:
      Run_Metadata: Metadata_Settings/Run_Metadata
      BamFiles: AddtoBam_VDJ/Annotated_Bam
      Generate_Bam: Bam_Settings/Generate_Bam
      Assay: Assay_Settings/Assay_RNA
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
    when: $(inputs.Generate_Bam == true && inputs.Cell_Calling_Data == 6 && (inputs.Assay == "WTA" || inputs.Assay == "Targeted"))
    out:
    - Bam
    - BamIndex
    - log
  MergeBAM_ATAC:
    run: MergeBAM.cwl
    in:
      Run_Metadata: Metadata_Settings/Run_Metadata
      BamFiles: AddtoBam_ATAC/Annotated_Bam
      Generate_Bam: Bam_Settings/Generate_Bam
      Assay: Assay_Settings/Assay_ATAC
    when: $(inputs.Generate_Bam == true && inputs.Assay == "ATAC")
    out:
    - Bam
    - BamIndex
    - log
  Metrics:
    requirements:
      ResourceRequirement:
        ramMin: 4000
    run: Metrics.cwl
    in:
      Metrics_tar:
        source: [ATAC_Compile_Results/Metrics_tar, GetDataTable/Metrics_tar]
        linkMerge: merge_flattened
        pickValue: first_non_null
      VDJ_Metrics_tar:
        source: VDJ_Compile_Results/vdjMetricsTar
      Run_Metadata: Metadata_Settings/Run_Metadata
      Cell_Type_Predictions:
        source: [GetDataTable/Cell_Type_Predictions, ATAC_Compile_Results/Cell_Type_Predictions]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Seq_Saturation_Plot_Json: CalcSeqSaturation/Seq_Saturation_Plot_Json
      SampleTag_Metrics_Tar: SampleTag/SampleTag_Metrics_Tar
    out:
    - Metrics_Summary
    - Metrics_Archive
    - Metrics_ATAC
    - Pipeline_Report_JSON
    - Pipeline_Report_HTML
    - output
  Peak_Annotation:
    run: PeakAnnotation.cwl
    in:
      Peaks_bed: QualCLAlign_ATAC/Peaks
      Assay: Assay_Settings/Assay_ATAC
      Run_Metadata: Metadata_Settings/Run_Metadata
      Gtf: CheckReference/GTF
    out:
      - Peak_Annotation_TSV
      - output
    when: $(inputs.Assay == "ATAC")  
  AtacMotifAnalysis:
    run: ATAC_Motif_Analysis.cwl
    in:
      Assay: Assay_Settings/Assay_ATAC
      Atac_Datatables: ATAC_Compile_Results/ATAC_Data_Tables
      Reference_FASTA: CheckReference/Reference_FASTA
      Transcription_Factor_Motif_PFM: CheckReference/Transcription_Factor_Motif_PFM
      Run_Metadata: Metadata_Settings/Run_Metadata
      Maximum_Threads: Maximum_Threads
    out:
      - ATAC_Motif_Data_Tables
      - ATAC_Peak_Seq
      - ATAC_Background_Peaks
      - ATAC_Motif_Positions
      - output
    when: $(inputs.Assay == "ATAC")  
  GenerateH5MU:
    run: GenerateH5MU.cwl
    in:
      GTF: CheckReference/GTF
      Reference_Genome_Contig_Lengths: QualCLAlign_ATAC/Reference_Genome_Contig_Lengths
      Transcription_Factor_Motif_PFM: CheckReference/Transcription_Factor_Motif_PFM
      Data_Tables: GetDataTable/Data_Tables
      Bioproduct_Stats: GetDataTable/Bioproduct_Stats
      Cell_Type_Experimental:
        source: [GetDataTable/Cell_Type_Predictions, ATAC_Compile_Results/Cell_Type_Predictions]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Putative_Cells_Origin: GetDataTable/Putative_Cells_Origin
      SampleTag_Calls: SampleTag/SampleTag_Calls
      SampleTag_CSVs: SampleTag/SampleTag_CSVs
      Dim_Reduction_Coord:
        source: [GetDataTable/Dim_Reduction_Coord, ATAC_Compile_Results/ATAC_Dim_Reduction_Coord]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Protein_Aggregates_Experimental: GetDataTable/Protein_Aggregates_Experimental
      VDJ_Per_Cell: VDJ_Compile_Results/vdjCellsDatatable
      Metrics_Summary: Metrics/Metrics_Summary
      Run_Metadata: Metadata_Settings/Run_Metadata
      Atac_Datatables: 
        source: [ATAC_Compile_Results/ATAC_Data_Tables, AtacMotifAnalysis/ATAC_Motif_Data_Tables]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Atac_Metrics: Metrics/Metrics_ATAC
      Peak_Annotation: Peak_Annotation/Peak_Annotation_TSV
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
      Atac_Peak_Seq: AtacMotifAnalysis/ATAC_Peak_Seq
      Atac_Background_Peaks: AtacMotifAnalysis/ATAC_Background_Peaks
      Atac_Motif_Positions: AtacMotifAnalysis/ATAC_Motif_Positions
    out:
      - H5MU
      - output
    when: $(inputs.Cell_Calling_Data != 6)
  GenerateSeurat:
    run: GenerateSeurat.cwl
    in:
      Data_Tables: GetDataTable/Data_Tables
      Bioproduct_Stats: GetDataTable/Bioproduct_Stats
      Cell_Type_Experimental: 
        source: [GetDataTable/Cell_Type_Predictions, ATAC_Compile_Results/Cell_Type_Predictions]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Putative_Cells_Origin: GetDataTable/Putative_Cells_Origin
      SampleTag_Calls: SampleTag/SampleTag_Calls
      SampleTag_CSVs: SampleTag/SampleTag_CSVs
      Dim_Reduction_Coord:
        source: [GetDataTable/Dim_Reduction_Coord, ATAC_Compile_Results/ATAC_Dim_Reduction_Coord]
        linkMerge: merge_flattened
        pickValue: all_non_null
      Protein_Aggregates_Experimental: GetDataTable/Protein_Aggregates_Experimental
      VDJ_Per_Cell: VDJ_Compile_Results/vdjCellsDatatable
      Run_Metadata: Metadata_Settings/Run_Metadata
      ATAC_Seurat: ATAC_Compile_Results/ATAC_Seurat_RDS
      Cell_Calling_Data: Putative_Cell_Calling_Settings/Cell_Calling_Data
      Atac_Motif_Datatables: AtacMotifAnalysis/ATAC_Motif_Data_Tables
      Atac_Peak_Seq: AtacMotifAnalysis/ATAC_Peak_Seq
      Atac_Background_Peaks: AtacMotifAnalysis/ATAC_Background_Peaks
      Atac_Motif_Positions: AtacMotifAnalysis/ATAC_Motif_Positions
      Transcription_Factor_Motif_PFM: CheckReference/Transcription_Factor_Motif_PFM
    out:
      - SeuratRDS
      - output
    when: $(inputs.Cell_Calling_Data != 6)
  BundleLogs:
    run: BundleLogs.cwl
    in:
      log_files:
        source:
        - CheckReference/output
        - GetDataTable/output
        - ATAC_Cell_by_Peak/output
        - ATAC_Compile_Results/output
        - Metrics/output
        - AddtoBam_RNA/output
        - AddtoBam_VDJ/output
        - AddtoBam_ATAC/output
        - AnnotateMolecules/output
        - MergeBAM_RNA/log
        - MergeBAM_VDJ/log
        - MergeBAM_ATAC/log
        - GenerateH5MU/output
        - GenerateSeurat/output
        - Peak_Annotation/output
        pickValue: all_non_null
        linkMerge: merge_flattened
    out:
    - logs_dir
  MergeMultiplex:
    run:
      cwlVersion: v1.2
      class: ExpressionTool
      requirements:
      - class: InlineJavascriptRequirement
      inputs:
        SampleTag_Files:
          type:
            items:
            - 'null'
            - File
            type: array
      outputs:
        Multiplex_out:
          type: File[]?
      expression: |-
        ${
          var fp_array = [];
          for (var i = 0; i < inputs.SampleTag_Files.length; i++) {
            var fp = inputs.SampleTag_Files[i];
            if (fp != null) {
              fp_array.push(fp);
            }
          }
          return({"Multiplex_out": fp_array});
        }
    in:
      SampleTag_Files:
        source: [SampleTag/SampleTag_CSVs, SampleTag/SampleTag_perTagZips]
        linkMerge: merge_flattened
    out:
      - Multiplex_out
  MergeATAC:
    run:
      cwlVersion: v1.2
      class: ExpressionTool
      requirements:
      - class: InlineJavascriptRequirement
      inputs:
        ATAC_Files:
          type:
            items:
            - 'null'
            - File
            type: array
      outputs:
        ATAC_out:
          type: File[]?
      expression: |-
        ${
          var fp_array = [];
          for (var i = 0; i < inputs.ATAC_Files.length; i++) {
            var fp = inputs.ATAC_Files[i];
            if (fp != null) {
              fp_array.push(fp);
            }
          }
          return({"ATAC_out": fp_array});
        }
    in:
      ATAC_Files:
        source: [QualCLAlign_ATAC/Fragments,       QualCLAlign_ATAC/Transposase_Sites,       QualCLAlign_ATAC/Peaks,
                 Peak_Annotation/Peak_Annotation_TSV, Metrics/Metrics_ATAC, ATAC_Compile_Results/ATAC_Data_Tables, 
                 AtacMotifAnalysis/ATAC_Motif_Data_Tables, AtacMotifAnalysis/ATAC_Motif_Positions]
        linkMerge: merge_flattened
    out:
      - ATAC_out
